﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Security.Cryptography;
using System.Text.RegularExpressions;
using System.Security.Cryptography.X509Certificates;
using System.IO;
using System.Windows.Forms;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;

namespace CRR_Example_Client
{
	public static class SecurityRSA
	{
		private static SHA256 _hashAlg = SHA256.Create();

		public static string OurPrivateKeyPEM = "";
		public static string TheirPublicCertificatePEM = "";

		public static RSACryptoServiceProvider PrivateKeyForSigningRequests = null;
		public static RSACryptoServiceProvider PublicKeyForValidatingAnswers = null;

		public static byte[] GetPem(string pem_key)
		{
			int start = 0, end = pem_key.Length;

			Match match_header = Regex.Match(pem_key, "-----BEGIN [ \\w]+-----[\\r]{0,1}\\n");
			Match match_footer = Regex.Match(pem_key, "-----END [ \\w]+-----");

			if (match_header.Success && match_footer.Success)
			{
				start = match_header.Index + match_header.Length;
				end = match_footer.Index;
			}

			string base64 = pem_key.Substring(start, (end - start));
			return Convert.FromBase64String(base64);
		}

		private static byte[] Normalize(byte[] in_array)
        {
			if (in_array == null) return null;

			int i = 0;
			for (; in_array[i] == 0 && i < in_array.Length; i++) ;
			if (i == 0) return in_array;

			int size = in_array.Length - i;
			byte[] buff = new byte[size];
			Array.Copy(in_array, in_array.Length - size, buff, 0, size);

			return buff;
		}

		private static int GetIntegerSize(BinaryReader binr)
		{
			byte bt = 0;
			byte lowbyte = 0x00;
			byte highbyte = 0x00;
			int count = 0;
			bt = binr.ReadByte();
			if (bt != 0x02)     //expect integer
				return 0;
			bt = binr.ReadByte();

			if (bt == 0x81)
				count = binr.ReadByte();    // data size in next byte
			else
				if (bt == 0x82)
			{
				highbyte = binr.ReadByte(); // data size in next 2 bytes
				lowbyte = binr.ReadByte();
				byte[] modint = { lowbyte, highbyte, 0x00, 0x00 };
				count = BitConverter.ToInt32(modint, 0);
			}
			else
			{
				count = bt;     // we already have the data size
			}

			while (binr.ReadByte() == 0x00)
			{   //remove high order zeros in data
				count -= 1;
			}
			binr.BaseStream.Seek(-1, SeekOrigin.Current);       //last ReadByte wasn't a removed zero, so back up a byte
			return count;
		}

		private static bool TLVDecode(byte[] InData, out byte[] tag, out int length, out byte[]data, out int total_length)
        {
			tag = null; length = 0; data = null; total_length = 0;
			int i = 0;
			// data has to be at least 2 bytes - one for tag, one for length
			if (InData == null || InData.Length < 2) return false;


			// tags can be one or more bytes long, depending on the most senior bits of the first byte, but we use only one byte tags here so skip that check
			tag = new byte[1];
			Array.Copy(InData, 0, tag, 0, 1);
			total_length += 1;
			i++;

			// data length of individual fields can be more than can fit in signle byte. If data is between 0 and 127 bytes long, the length is directly written as single byte.
			// If it is 128 or longer, then the first byte has format of 0x8X, where X is the count of bytes that follow up to 4 bytes for a 32 bit unsigned integer. The rest is an integer of X bytes with most senior byte first
			if ((InData[i] & 0x80) == 0x80)
            {
				int count = InData[i] & 0x0F;
				if (count > 4) return false;
				byte[] modint = { 0x00, 0x00, 0x00, 0x00 };
				for (int j = 0; j < count; j++)
                {
					modint[j] = InData[i + count - j];
				}
				length = BitConverter.ToInt32(modint, 0);
				total_length += count + 1;
				i += count + 1;
			}
			else
            {
				length = InData[i];
				total_length += 1;
				i += 1;
			}

			data = new byte[length];
			Array.Copy(InData, i, data, 0, length);

			total_length += length;

			return true;
		}

		public static RSACryptoServiceProvider DecodeRSAPrivateKey(byte[] privkey)
		{
			byte[] MODULUS, E, D, P, Q, DP, DQ, IQ;
			int key_size = 0;

			byte[] tag, data;
			int len, total_len;

			try
			{
				// There are two primary formats that we decode. PKCS#1 and PKCS#8 (not encrypted)
				// Both are a collection of Tag-Length-Value (TLV) structures nested one in another in a specific order

				// Both formats start with container with tag 30

				if (!TLVDecode(privkey, out tag, out len, out data, out total_len))
					return null;

				if (tag[0] != 0x30)
					return null;

				byte[] container = data;
				int index = 0;

				// First element in container is with tag 02 and data "00"
				if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
					return null;

				if (tag[0] != 0x02 || len != 1 || data[0] != 0x00)
					return null;

				index += total_len;

				// after that they start to differ
				if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
					return null;

				// If tag is again 02, then this is format PKCS#1. If tag is again 30, then it is PKCS#8
				if (tag[0] == 0x30)
				{
					index += total_len;

					// This is PKCS#8 format and we need to skip some more tags.
					if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
						return null;

					// There should be a tag 04, that contains another tag 30, that contains the key data
					if (tag[0] != 0x04)
						return null;

					container = data;
					index = 0;

					if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
						return null;

					if (tag[0] != 0x30)
						return null;

					container = data;
					index = 0;

					// And first element of the last 30 container is again tag 02 with data "00"

					if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
						return null;

					if (tag[0] != 0x02 || len != 1 || data[0] != 0x00)
						return null;

					index += total_len;

					// Read again to match where we would be if format was PKCS#1
					if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
						return null;
				}

				// Now we should have 8 consecutive 02 tags, which contain the Private key elements in order MODULUS, E(public exponent), D(private exponent), P, Q, DP, DQ, IQ
				if (tag[0] != 0x02)
					return null;

				MODULUS = data;
				// the biggest element here should define the size of the key object we make later
				if (len > key_size) key_size = len;
				index += total_len;

				if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
					return null;
				if (tag[0] != 0x02)
					return null;

				E = data;
				if (len > key_size) key_size = len;
				index += total_len;

				if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
					return null;
				if (tag[0] != 0x02)
					return null;

				D = data;
				if (len > key_size) key_size = len;
				index += total_len;

				if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
					return null;
				if (tag[0] != 0x02)
					return null;

				P = data;
				if (len > key_size) key_size = len;
				index += total_len;

				if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
					return null;
				if (tag[0] != 0x02)
					return null;

				Q = data;
				if (len > key_size) key_size = len;
				index += total_len;

				if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
					return null;
				if (tag[0] != 0x02)
					return null;

				DP = data;
				if (len > key_size) key_size = len;
				index += total_len;

				if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
					return null;
				if (tag[0] != 0x02)
					return null;

				DQ = data;
				if (len > key_size) key_size = len;
				index += total_len;

				if (!TLVDecode(container.Skip(index).ToArray(), out tag, out len, out data, out total_len))
					return null;
				if (tag[0] != 0x02)
					return null;

				IQ = data;
				if (len > key_size) key_size = len;
				index += total_len;

				// Some fields start with a leading 00. C# doesn't like that, so cut that leading zero

				MODULUS = Normalize(MODULUS);
				E = Normalize(E);
				D = Normalize(D);
				P = Normalize(P);
				Q = Normalize(Q);
				DP = Normalize(DP);
				DQ = Normalize(DQ);
				IQ = Normalize(IQ);

				// ------- create RSACryptoServiceProvider instance and initialize with public key -----
				CspParameters cp = new CspParameters();
				cp.Flags = CspProviderFlags.UseMachineKeyStore;
				cp.KeyContainerName = "CRRKeys";

				RSACryptoServiceProvider RSA = new RSACryptoServiceProvider(key_size * 8, cp);
				RSAParameters RSAparams = new RSAParameters();

				RSAparams.Modulus = MODULUS;
				RSAparams.Exponent = E;
				RSAparams.D = D;
				RSAparams.P = P;
				RSAparams.Q = Q;
				RSAparams.DP = DP;
				RSAparams.DQ = DQ;
				RSAparams.InverseQ = IQ;

				RSA.ImportParameters(RSAparams);
				return RSA;
			}
			catch //(Exception ex)
			{
				return null;
			}
		}

		public static byte[] HashAndSignData(byte[] binDataToSign)
		{
			if (PrivateKeyForSigningRequests == null) throw new Exception("Private key not initialized");

			byte[] binSignature = null;
			binSignature = PrivateKeyForSigningRequests.SignData(binDataToSign, _hashAlg);
			return binSignature;
		}

		public static bool HashAndVerifyData(byte[] binDataToVerify, byte[] binSignature)
		{
			if (PublicKeyForValidatingAnswers == null) throw new Exception("Public key not initialized");

			bool bVerified = false;
			bVerified = PublicKeyForValidatingAnswers.VerifyData(binDataToVerify, _hashAlg, binSignature);
			return bVerified;
		}
	}
}
